const { EOL } = require("os");
const constants = require("./constants.js");
const CHILD_PROCESS_MESSAGES = constants.CHILD_PROCESS_MESSAGES;
const ERROR_CODES = constants.ERROR_CODES;

let runnerShell = null;
let loadedSuccessfully = false;

// handle Uncaught Exceptions
const handleProcError = err => {
	let type;
	let error;
	let code;

	if (loadedSuccessfully) {
		type = CHILD_PROCESS_MESSAGES.UNCAUGHT_EXCEPTION;
		code = ERROR_CODES.UNCAUGHT_EXCEPTION;
		error = {
			message: err.message,
			details: err.details,
			code: code,
			stack: err.stack
		};
	} else {
		type = CHILD_PROCESS_MESSAGES.SETUP_FAIL;
		code = ERROR_CODES.PROCESS_SETUP_FAIL;
		error = {
			message: "Unable to launch process",
			details: err.message,
			code: code,
			stack: err.stack
		};
	}
	process.send({
		type,
		payload: {
			error
		}
	});

	if (runnerShell) runnerShell.shutdown(code);
};

process.on("uncaughtException", handleProcError);
process.on("unhandledRejection", handleProcError);

const NSProcessShell = require("./nsProcessShell.js");
runnerShell = new NSProcessShell();

// listen for TERM signal .e.g. kill
process.on("SIGTERM", runnerShell.shutdown);
// listen for INT signal e.g. Ctrl-C
process.on("SIGINT", runnerShell.shutdown);


// Shutdown on RunnerShell Errors
runnerShell.on("error", (err) => {
	process.send({
		type: CHILD_PROCESS_MESSAGES.UNCAUGHT_EXCEPTION,
		payload: {
			error: {
				message: err.message,
				details: err.details,
				code: err.code,
				stack: err.stack
			}
		}
	});

	runnerShell.shutdown(ERROR_CODES.UNCAUGHT_EXCEPTION);
});

runnerShell.setup()
	.then(() => {

		// Advertise subset of process shell API
		runnerShell.registerPublicAPI();

		// write initial newline char
		process.stdout.write(EOL);
		process.stderr.write(EOL);

		// Kick off things and load user script
		require(process.env.SCRIPT_PATH);
		loadedSuccessfully = true;

	})
	.catch((err) => {
		process.send({
			type: CHILD_PROCESS_MESSAGES.SETUP_FAIL,
			payload: {
				error: {
					message: "Unable to launch process",
					details: err.message,
					code: ERROR_CODES.PROCESS_SETUP_FAIL,
					stack: err.stack
				}
			}
		});
		return runnerShell.shutdown(ERROR_CODES.PROCESS_SETUP_FAIL);
	});


